/************************************************************************
 * NAME:	fs-ados33.c
 *
 * DESCR:	Implements the Apple 2 Dos 3.3 file system.
 *
 *
 * NOTES:	
 ************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include "getopt.h"

#include "standard.h"
#include "floppy.h"
#include "fs-ados33.h"


/************************************************************************
 * NAME:	fs_ados33_init() and fs_ados33_cleanup()
 *
 * DESCR:	Initializes/cleans-up the ADOS33 file system structures.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_ados33_init(struct floppy *floppy, struct fs_ados33 *fs)
{
    extern int floppy_read_sector(struct floppy *, int, int, int, char *);
    extern int floppy_write_sector(struct floppy *, int, int, int, char *);

    if (!ados33_init(&fs->ados33fs,floppy, (adosreadfn)floppy_read_sector,
		                           (adoswritefn)floppy_write_sector)) {
	return(FALSE);
    }

    return (ados33_settings(&fs->ados33fs,0,5));
}

int
fs_ados33_cleanup(struct fs_ados33 *fs)
{
    return(ados33_cleanup(&fs->ados33fs));
}

/************************************************************************
 * NAME:	fs_ados33_report()
 *
 * DESCR:	Generates a report on the filesystem structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_ados33_report(struct fs_ados33 *fs, int verbosity)
{
    ados33_dir_report(&fs->ados33fs,verbosity);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_ados33_add()
 *
 * DESCR:	Adds the referenced file to the ADOS33 image.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_ados33_add(struct fs_ados33 *fs, char *name, int unusedfd)
{
    int			 i;
    struct ados_file	*output;
    int			 input;
    char		 buffer[BUFSIZ];
    ados_file_entry	*file_entry;

    output = ados_file_new(&fs->ados33fs, name);
    if (output == NULL) {
	return(FALSE);		/* file exists or no spacae	*/
    }

    if ((input = open(name,O_RDONLY)) < 0) {
	return(FALSE);
    }

    for (;;) {
	i = read(input,buffer,BUFSIZ);
	if (i==0)
	    break;
	i -= ados_file_write(output,buffer,i);
    }

    ados_file_close(output);
    close(input);

    if (!ados_sync(&fs->ados33fs)) {
	return(FALSE);
    }
  
    return(TRUE);
}

/************************************************************************
 * NAME:	fs_ados33_del()
 *
 * DESCR:	Deletes the named file from the filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
fs_ados33_del(struct fs_ados33 *fs, char *name)
{
    if (!ados_file_delete(&fs->ados33fs,name)) {
	return(FALSE);
    }

    if (!ados_sync(&fs->ados33fs)) {
	return(FALSE);
    }
  
    return(TRUE);
}

/************************************************************************
 * NAME:	fs_ados33_extract()
 *
 * DESCR:	Extracts the named file from the filesystem, sending it
 *		out
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the ADOS routines do NOT do much file conversion, which
 *		  is different from the capabilities of Ciderpress.  Here
 *		  are the things it does that you should know about:
 *		   - all files with leading binary data maintain that data
 *			when extracted, meaning that the resultant file
 *			may be longer than as if extracted by Ciderpres
 *		   - text files are NOT converted, meaning that high-bit
 *			set files won't have the high bit stripped (making
 *			the text unreadable)
 *		   - basic files are NOT converted to something more readable
 ************************************************************************/
int
fs_ados33_extract(struct fs_ados33 *fs, char *name, int fd)
{
    int			 i;
    struct ados_file	*input;
    int			 output;
    char		 buffer[BUFSIZ];
    ados_file_entry	*file_entry;

    input = ados_file_open(&fs->ados33fs, name);
    if (input == NULL) {
	return(FALSE);
    }

    if ((output = open(name,O_CREAT|O_WRONLY|O_TRUNC,0777)) < 0) {
	return(FALSE);
    }

    for (;;) {
	i = ados_file_read(input,buffer,BUFSIZ);
	if (i==0) 
	    break;
	write(output,buffer,i);
    }

    ados_file_close(input);
    close(output);

    return(TRUE);
}

/************************************************************************
 * NAME:	fs_ados33_description()
 *
 * DESCR:	Returns the description of this type of file system.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
fs_ados33_description(void)
{
    return("Apple ][ Dos 3.3");
}


/************************************************************************
 * NAME:	fs_ados33_compress()
 *
 * DESCR:
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 ************************************************************************/
int
fs_ados33_compress(struct fs_ados33 *fs)
{
}
